/*
 *  Arnold emulator (c) Copyright, Kevin Thacker 1995-2015
 *
 *  This file is part of the Arnold emulator source code distribution.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include "../cpc.h"
#include "../cpcglob.h"

#include "timer.h"

#define MAX_TIMER_STACK 256
static int TimerStackNum = 0;
static TIMER *TimerStack[MAX_TIMER_STACK];

static TIMER *pFirstTimer = NULL;

static BOOL TimerRequireFullRefresh = FALSE;

BOOL Timers_FullRefreshInGUI()
{
    return TimerRequireFullRefresh;
}

void Timers_SetFullRefreshInGUI(BOOL bState)
{
	TimerRequireFullRefresh = bState;
}

TIMER *Timers_GetFirst()
{
	return pFirstTimer;
}

TIMER *Timers_GetNext(TIMER *pTimer)
{
	return pTimer->pNext;
}

BOOL Timer_RequireRefreshInGUI(TIMER *pTimer)
{
    return pTimer->RequireRefreshInGUI;
}

void Timer_SetRequireRefreshInGUI(TIMER *pTimer, BOOL bState)
{
    pTimer->RequireRefreshInGUI = bState;
}

#if 0
BOOL Timer_Valid(TIMER *pTimerWanted)
{
	TIMER *pTimer;

	pTimer = pFirstTimer;

	while (pTimer != NULL)
	{
		if (pTimer == pTimerWanted)
			return TRUE;

		pTimer = pTimer->pNext;
	}
	return FALSE;
}
#endif

/* add a breakpoint */
TIMER *Timer_AddTimerI(TIMER_TYPE Type,int Address)
{
	TIMER *pTimer;

	pTimer = (TIMER *)malloc(sizeof(TIMER));

	if (pTimer!=NULL)
	{
		memset(pTimer, 0, sizeof(TIMER));

		pTimer->Address = Address;
		pTimer->nTimings = 0;
		pTimer->bEnabled = TRUE;
		pTimer->pNext = pFirstTimer;
		pTimer->Type = Type;
		pTimer->bFirstHit = TRUE;
		pTimer->RequireRefreshInGUI = TRUE;
		Timers_SetFullRefreshInGUI(TRUE);
		pFirstTimer = pTimer;
	}
	return pTimer;
}

void Timer_Reset(TIMER *pTimer)
{
	pTimer->nTimings = 0;
	pTimer->bFirstHit = TRUE;
}

/* add a breakpoint */
TIMER *Timers_AddTimer(TIMER_TYPE nType,int Address)
{
	return Timer_AddTimerI(nType,Address);
}

/* remove an existing breakpoint */
void    Timers_RemoveTimer(TIMER *pTimer)
{
	if (pTimer==pFirstTimer)
	{
		pFirstTimer = pTimer->pNext;
	}
	else
	{
		TIMER *pCurrent = pFirstTimer;
		while (pCurrent!=NULL)
		{
			if (pCurrent->pNext==pTimer)
			{
				pCurrent->pNext = pTimer->pNext;
				break;
			}

			pCurrent = pCurrent->pNext;
		}

	}
	free(pTimer);
	Timers_SetFullRefreshInGUI(TRUE);
}

/* free all breakpoints */
void	Timer_Free(void)
{
	TIMER *pTimer;

	pTimer = pFirstTimer;

	while (pTimer!=NULL)
	{
		TIMER *pNext;

		pNext = pTimer->pNext;

		free(pTimer);

		pTimer = pNext;
	}

	pFirstTimer = NULL;
	Timers_SetFullRefreshInGUI(TRUE);
}

void Timers_SetEnabled(TIMER *pTimer, BOOL bEnable)
{
	pTimer->bEnabled = bEnable;
	pTimer->RequireRefreshInGUI = TRUE;
}



void Timers_Update(int nAddress)
{
	int HitTime = CPC_GetNopCount();

	TIMER *pTimer;

	pTimer = pFirstTimer;

	while (pTimer != NULL)
	{
		if (pTimer->Address == nAddress)
		{
			if (pTimer->Type == TIMER_HIT)
			{
				if (pTimer->bFirstHit)
				{
					pTimer->LastHitTime = HitTime;
					pTimer->bFirstHit = FALSE;
				}
				else
				{
					int Delta = HitTime - pTimer->LastHitTime;
					pTimer->TimingHistory[pTimer->nTimingPosition] = Delta;
					pTimer->nTimingPosition++;
					if (pTimer->nTimingPosition >= MAX_TIMING_HISTORY)
					{
						pTimer->nTimingPosition = 0;
					}
					pTimer->nTimings++;
					if (pTimer->nTimings >= MAX_TIMING_HISTORY)
					{
						pTimer->nTimings = MAX_TIMING_HISTORY;
					}
					// store last hit time.
					pTimer->LastHitTime = HitTime;
				}
			}
			else if (pTimer->Type==TIMER_START)
			{
				pTimer->TimingHistory[pTimer->nTimingPosition] = HitTime;
				pTimer->nTimingPosition++;
				if (pTimer->nTimingPosition >= MAX_TIMING_HISTORY)
				{
					pTimer->nTimingPosition = 0;
				}
				pTimer->nTimings++;
				if (pTimer->nTimings >= MAX_TIMING_HISTORY)
				{
					pTimer->nTimings = MAX_TIMING_HISTORY;
				}
				/* push on stack if we're not already on there */
				if (TimerStackNum != 0)
				{
					if (TimerStack[TimerStackNum - 1]!=pTimer)
					{
						TimerStack[TimerStackNum] = pTimer;
						TimerStackNum++;
					}
				}
				else
				{
					TimerStack[TimerStackNum] = pTimer;
					TimerStackNum++;
				}
			}
			else if (pTimer->Type == TIMER_END)
			{
				TIMER *pTimerStart = NULL;
			
				/* pop off stack */
				if (TimerStackNum != 0)
				{
					pTimerStart = TimerStack[TimerStackNum - 1];
					TimerStackNum = 0;
				}

				if (pTimerStart)
				{
					int nLastTime = pTimerStart->nTimingPosition-1;
					if (nLastTime < 0)
					{
						nLastTime = MAX_TIMING_HISTORY - 1;
					}

					int StartTime = pTimerStart->TimingHistory[nLastTime];
					pTimer->TimingHistory[pTimer->nTimingPosition] = HitTime-StartTime;
					pTimer->nTimingPosition++;
					if (pTimer->nTimingPosition >= MAX_TIMING_HISTORY)
					{
						pTimer->nTimingPosition = 0;
					}
					pTimer->nTimings++;
					if (pTimer->nTimings >= MAX_TIMING_HISTORY)
					{
						pTimer->nTimings = MAX_TIMING_HISTORY;
					}
				}
			}
		}
		pTimer = pTimer->pNext;
	}
}

BOOL Timer_GetStats(TIMER *pTimer, TIMER_STATS *pStats, BOOL bFPS)
{
	if (pTimer->nTimings == 0 || pTimer->Type ==TIMER_START)
	{
		return FALSE;
	}

	{
		int i;
		pStats->nMin = pTimer->TimingHistory[0];
		pStats->nMax = pStats->nMin;
		pStats->nAvg = pStats->nMin;
		for (i = 1; i < pTimer->nTimings; i++)
		{
			int nValue = pTimer->TimingHistory[i];
			if (nValue < pStats->nMin)
			{
				pStats->nMin = nValue;
			}
			if (nValue > pStats->nMax)
			{
				pStats->nMax = nValue;
			}
			pStats->nAvg += nValue;
		}
		pStats->nAvg /= pTimer->nTimings;

		if (bFPS)
		{
			// min is best, max is worst.
			// so for fps we need to reverse it so that min is worst and max is best.
			int nMin = pStats->nMin;
			int nMax = pStats->nMax;

			if (nMax != 0)
			{
				pStats->nMin = (19968 * 50) / nMax;
			}
			else
			{
				pStats->nMin = 0;
			}
			if (nMin != 0)
			{
				pStats->nMax = (19968 * 50) / nMin;
			}
			else
			{
				pStats->nMax = 0;
			}
			// avg is still avg in fps
			if (pStats->nAvg != 0)
			{
				pStats->nAvg = (19968 * 50) / pStats->nAvg;
			}
		}
	}
	return TRUE;
}
